  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Wed Nov 17 17:25:55 2004
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief A brief description of this module
   *
   *  \version CRYS_AES.c#1:csrc:6
   *  \author adams
   *  \remarks Copyright (C) 2004 by Discretix Technologies Ltd.
   *           All Rights reserved
   */



/************* Include Files ****************/

#include <linux/module.h>
#include "CRYS_RND_error.h" 
#include "CRYS_RND.h"
#include "error.h"
#include "host_op_code.h"
#include "SEPDriver.h"

/************************ Defines ******************************/

/* canceling the lint warning:
   Use of goto is deprecated */
/*lint --e{801} */

/************************ Enums ******************************/


/************************ macros ******************************/

/************************ Global Data ******************************/



/************* Private function prototype ****************/


/************************ Public Functions ******************************/

/* ------------------------------------------------------------
 * @brief The CRYS_RND_GenerateVector function generates a random vector using the FIPS-PUB
 *        186-2 standard appendex 3.
 *
 *        The random function is based on the HASH SHA1 and executes the follow:
 *
 *        1. loads the secret key stored internally in a globla variable.
 *        2. executes the following steps as the number of hash result blocks required (RndSize/20).
 *           a. executes a SHA1 hash on the current key ( if the key is less then 512 bytes then
 *              add zeros ).
 *           b. store the result to the data  
 *           c. the next key is 1 ^ key ^ result. this is the new key.     
 *
 * @RndSize[in] - The size of random vectore that is required.
 *
 * @Output_ptr[in,out] The output vector.
 *
 * @return CRYSError_t - On success CRYS_OK is returned, on failure a
 *                        value MODULE_* as defined in ...
 */
CEXPORT_C CRYSError_t CRYS_RND_GenerateVector(DxUint16_t  RndSize,DxUint8_t  *Output_ptr )
{
  /* error identifier definition */
  CRYSError_t   Error;
  
  /* offset */
  DxUint32_t    sramOffset;
  
  /* read parameter */
  DxUint32_t    messageParam[2];
  
  /* max length */
  DxUint32_t    maxLength;

  /* max length */
  DxUint32_t    numOfLoops;

  /* max length */
  DxUint32_t    tempLen;
  /*---------------------------------------
      CODE
  ----------------------------------------*/
  
  #ifndef CRYS_NO_HASH_SUPPORT                                  
 
  /* ................... checking parameters validity ..................... */
   
  if( Output_ptr == DX_NULL )
  {
    Error = CRYS_RND_DATA_OUT_POINTER_INVALID_ERROR;
    goto end_function;
  }
 
  /* ................... initializing local variables ..................... */
   
  Error = CRYS_OK;
  
   /* lock access to the SEP */
   Error = SEPDriver_Lock();
   
   if(Error != DX_OK)
   {
       goto end_function;
   }
  
	numOfLoops = RndSize / 0x1000;
	if(RndSize % 0x1000 )
	{
		numOfLoops = numOfLoops +1;
	}

	while(numOfLoops)
	{
		  /*----------------------------
		      start sending message to SEP 
		  -----------------------------*/
		  sramOffset = 0;
		   
		  /* start the message */
		  SEPDriver_StartMessage(&sramOffset);
		  
			if(RndSize > 0x1000)
			{
				tempLen = 0x1000;
			}
			else
			{
				tempLen = RndSize;		
			}
		  /* prepare params */
		  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_RND_GENERAT_VECTOR_OP_CODE;
		  messageParam[1] = tempLen;
		  
		  /* send params */
		  Error = SEPDriver_WriteParamater((DxUint32_t)messageParam ,
		                           sizeof(DxUint32_t) * 2,
		                           sizeof(DxUint32_t) * 2, 
		                           &sramOffset , 
		                           DX_FALSE);
		  if(Error != DX_OK)
		  {
		      goto end_function_unlock;
		  }
		  
		  SEPDriver_EndMessage(sramOffset);
		            
		  /* wait for the response */
		  Error = SEPDriver_POLL_FOR_REPONSE();
		  if(Error != DX_OK)
		  {
		  	goto end_function_unlock;
		  }
		   
		  /*-------------------
		    start reading message from the SEP 
		  ---------------------*/
		   
		  /* start the message */
		  Error = SEPDriver_StartIncomingMessage(&sramOffset);
		  if(Error != DX_OK)
		  {
		  	goto end_function_unlock;
		  }
		   
		  /* read opcode + status */
		  Error = SEPDriver_ReadParamater((DxUint32_t)messageParam ,
		                          sizeof(DxUint32_t) * 2,
		                          sizeof(DxUint32_t) * 2, 
		                          &sramOffset , 
		                          DX_FALSE);
		  if(Error != DX_OK)
		  {
		      goto end_function_unlock;
		  }
		   
		  /* check the opcode */
		  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_RND_GENERAT_VECTOR_OP_CODE)
		  {
		    Error = DX_WRONG_OPCODE_FROM_SEP_ERR;
		    goto end_function_unlock;
		  }
		   
		  /* read the status */
		  if(messageParam[1] != CRYS_OK)
		  {
		    Error = messageParam[1];
		    goto end_function_unlock;
		  }
		   
		  /* read the vector */
		  maxLength = ((tempLen + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
		  Error = SEPDriver_ReadParamater((DxUint32_t)Output_ptr,
		                           tempLen,
		                           maxLength,
		                           &sramOffset,
		                           DX_TRUE);
		  if(Error != DX_OK)
		  {
		      goto end_function_unlock;
		  }
		                                           
		  /* ...................... end of function ................................ */   
		Output_ptr = Output_ptr + tempLen;
		RndSize = RndSize - tempLen;
		
		numOfLoops--;
	} /* while(numOfLoops) */

end_function_unlock:   

  /* lock access to the SEP */
  SEPDriver_Unlock();
                      
end_function:

  return Error;
   
  #endif /*CRYS_NO_HASH_SUPPORT*/
   
}/* END OF CRYS_RND_GenerateVector */

EXPORT_SYMBOL(CRYS_RND_GenerateVector);


/**********************************************************************************************************/
/**
 * @brief The CRYS_RND_ResetSeed resets the SEED generated by the low level ( hardware or OPerating system
 *        service on software ).          
 *
 * @return CRYSError_t - On success CRYS_OK is returned, on failure a
 *                        value MODULE_* as defined in ...
 */
CEXPORT_C CRYSError_t CRYS_RND_ResetSeed( void )
{
  /* error identifier definition */
  CRYSError_t   Error;
  
  /* offset */
  DxUint32_t    sramOffset;
  
  /* read parameter */
  DxUint32_t    messageParam[2];

  /*--------------------------------
      CODE
  ----------------------------------*/
  
  Error = CRYS_OK;
  
   /* lock access to the SEP */
   Error = SEPDriver_Lock();
   
   if(Error != DX_OK)
   {
       goto end_function_no_unlock;
   }
  
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare params */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_RND_RESET_SEED_OP_CODE;
  
  /* send params */
  Error = SEPDriver_WriteParamater((DxUint32_t)messageParam ,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(Error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  Error = SEPDriver_POLL_FOR_REPONSE();
  if(Error != DX_OK)
  {
  	goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  Error = SEPDriver_StartIncomingMessage(&sramOffset);
  if(Error != DX_OK)
  {
  	goto end_function_unlock;
  }
   
  /* read opcode + status */
  Error = SEPDriver_ReadParamater((DxUint32_t)messageParam ,
                          sizeof(DxUint32_t) * 2,
                          sizeof(DxUint32_t) * 2,
                          &sramOffset , 
                          DX_FALSE);
  if(Error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_RND_RESET_SEED_OP_CODE)
  {
    Error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    goto end_function_unlock;
  }
   
  /* check the status */
  if(messageParam[1] != CRYS_OK)
  {
    Error = messageParam[1];
    goto end_function_unlock;
  }
                                           
  /* ...................... end of function ................................ */   

end_function_unlock:   

  /* lock access to the SEP */
  SEPDriver_Unlock();

end_function_no_unlock:

  return Error;
  
}/* END OF CRYS_RND_ResetSeed */

EXPORT_SYMBOL(CRYS_RND_ResetSeed);



/**********************************************************************************************************/
/* 
 * @brief The CRYS_RND_GenerateVector function generates a random vector Rand in range:
 *            1 < RandVect < MaxVect,   using the FIPS-PUB 186-2 standard appendex 3 :
 *        The function performs the following: 
 *        1.  Calls the CRYS_RND_GenerateVector() function for generating random vector 
 *            RndVect of size in bytes.
 *        2.  Zeroes extra bits of RndVect.
 *        3.  Compares RndVect with 1 and MaxVect. If condition  1 < RandVect < MaxVect
 *            is not satisfied goues to step 1.
 *
 * @RndSize[in] - The size of random vectore that is required.
 * @MaxVect_ptr[in]  - The pointer to vector defines high limit of random vector.
 * @RndVect_ptr[in,out] The output vector.
 * @return CRYSError_t - On success CRYS_OK is returned, on failure a
 *                        value MODULE_* as defined in ...
 */
CEXPORT_C CRYSError_t CRYS_RND_GenerateVectorInRange(DxUint32_t   RndSizeInBits, 
											                               DxUint8_t*   MaxVect_ptr, 
											                               DxUint8_t*   RndVect_ptr )
{
  /* error */
  CRYSError_t   Error;
  
   /* offset */
  DxUint32_t    sramOffset;
  
  /* read parameter */
  DxUint32_t    messageParam[2];
  
  /* max length */
  DxUint32_t    maxLength;
  
  /*----------------------------------------
      CODE
  ------------------------------------------*/
  
	Error = CRYS_OK;
	
   /* lock access to the SEP */
   Error = SEPDriver_Lock();
   
   if(Error != DX_OK)
   {
       goto end_function_no_unlock;
   }

  
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare params */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_RND_GENERAT_VECTOR_IN_RANGE_OP_CODE;
  messageParam[1] = RndSizeInBits;
  
  /* send params */
  Error = SEPDriver_WriteParamater((DxUint32_t)messageParam , 
                           sizeof(DxUint32_t) * 2,
                           sizeof(DxUint32_t) * 2,
                           &sramOffset , 
                           DX_FALSE);
  if(Error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* send boundary vector */
  maxLength = (( ((RndSizeInBits + 7) / 8) + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  Error = SEPDriver_WriteParamater((DxUint32_t)MaxVect_ptr , 
                            (RndSizeInBits + 7) / 8 ,
                             maxLength, 
                             &sramOffset , 
                             DX_FALSE);
  if(Error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  Error = SEPDriver_POLL_FOR_REPONSE();
  if(Error != DX_OK)
  {
  	goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  Error = SEPDriver_StartIncomingMessage(&sramOffset);
  if(Error != DX_OK)
  {
  	goto end_function_unlock;
  }
   
  /* read opcode + status  */
  Error = SEPDriver_ReadParamater((DxUint32_t)messageParam , 
                          sizeof(DxUint32_t) * 2,
                          sizeof(DxUint32_t) * 2,
                          &sramOffset , 
                          DX_FALSE);
  if(Error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_RND_GENERAT_VECTOR_IN_RANGE_OP_CODE)
  {
    Error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    goto end_function_unlock;
  }
   
  /* check the status */
  if(messageParam[1] != CRYS_OK)
  {
    Error = messageParam[1];
    goto end_function_unlock;
  }
   
  /* read the context */
  maxLength = (( ((RndSizeInBits + 7) / 8) + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  Error = SEPDriver_ReadParamater((DxUint32_t)RndVect_ptr,
                           (RndSizeInBits + 7) / 8,
                            maxLength,
                            &sramOffset,
                            DX_TRUE);
  if(Error != DX_OK)
  {
      goto end_function_unlock;
  }
                                           
  /* ...................... end of function ................................ */   

end_function_unlock:   

  /* lock access to the SEP */
  SEPDriver_Unlock();
  
end_function_no_unlock:

  return Error;
    
} /* End of CRYS_RND_GenerateVectorInRange function */

EXPORT_SYMBOL(CRYS_RND_GenerateVectorInRange);

